/*
 * Copyright (c) 2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 * \file    DM8127_i2c.h
 *
 * \brief   Defines the module register defines for I2C
 *
 * This header file contains the Macros definitions and the
 * register overlay structure for DM8127 I2C controller.
 *
 * \author  0043
 *
 * \version 01a,13aug,2010 Created.
 */

#ifndef _DM8127_aic3104_H_
#define _DM8127_aic3104_H_

#ifdef __cplusplus
 extern "C" {
#endif


/*
 *====================
 * Includes
 *====================
 */
#include "DM8127_types.h"

/**
@addtogroup DM8127 aic3104 Audio Codec
@{
*/

/*
 *====================
 * Defines
 *====================
*/
#define aic3104_I2C_ADDR              (0x18)    // I2C address
/**< I2C Slave address of the aic3104 Audio Codec chip */

#define aic3104_I2C_BUS_NUM_0		  (0u)
/**< Bus Number where aic3104 is connected on the EVM */

#define aic3104_I2C_BUS_NUM_1		  (1u)
/**< Bus Number where aic3104 is connected on the EVM */

#define aic3104_I2C_BUS_NUM_2		  (2u)
/**< Bus Number where aic3104 is connected on the EVM */

#define aic3104_I2C_BUS_NUM_3		  (3u)
/**< Bus Number where aic3104 is connected on the EVM */


#define aic3104_REG_ADDR_MASK	      (0x007F);
/**< Mask for AIC316 internal register offsets */

 /*
  * MCASP0 0x48038000
  * MCASP0 0x4803C000
  * MCASP2 0x48050000
  * MCASP3 0x4A1A2000
  * MCASP4 0x4A1A8000
  * MCASP5 0x4A1AE000
  */
#define 	DM8127_EVM_aic3104_MCASP  (0x48050000)
/**< Base address of the McASP being used for aic3104 on Base Board */

#define aic3104_PAGESELECT        0
/**< aic3104 Page Select */

#define aic3104_RESET             1
/**< aic3104 reset */
/*
 *====================
 * Function declarations
 *====================
 */

 /***
  * \brief: Routine to set the register value in the aic3104 Codec
  *
  * \bDescr: This routine initializes the I2C instance 0. We are not configuring
  * \n       the interrupts here as the code is intended to carry out all I/O
  * \n       operations in polled mode.
  *
  * \param:  u8RegNum		[IN]	Bus Instance number
  * \n       u8RegVal		[IN]	Value to be written
  *
  * \return:   SUCCESS    on success
  * \n         FAILED     if any error
  */

extern STATUS aic3104RegSet
(
	UINT8	u8RegNum,
	UINT8  	u8RegVal
);

/***
 * \brief: Routine to configure the Headphone Out and play sine tone
 *
 * \brief: This routine initializes the aic3104 Audio Codec on the
 *        base board and plays a sine tone on the head-phone.
 *
 * \param : u8McaspNum	[IN]	Instance of McASP number
 *
 * \return: SUCCESS  on success
 *         FAILED   if any error
 */

STATUS aic3104_headphone_out
(
	UINT8    u8McaspNum
);

/** **************************************************************************
 * \n \brief Routine to configure the Headphone Out and play sine tone
 *
 * This routine initializes the aic3104 Audio Codec on the base board and
 * plays a sine tone on the head-phone.
 *
 * \param u8McaspNum	[IN]	Instance of McASP number
 *
 * \return
 * \n      return SUCCESS for success  - Description
 * \n      return FAILED for error   - Description
 */
STATUS aic3104_lineout_out
(
	UINT8    u8McaspNum
);

/** **************************************************************************
 * \n \brief Routine to configure aic3104 to loop Line-In to Line-Out
 *
 * This routine initializes the aic3104 Audio Codec on the base board to take
 * input from the Line-In and play it back to Line-Out.
 *
 * \param u8McaspNum	[IN]	Instance of McASP number
 *
 * \return
 * \n      return SUCCESS for success  - Description
 * \n      return FAILED for error   - Description
 */
STATUS aic3104_LineInLineOut_Loop
(
	UINT8    u8McaspNum
);

/***
 * \brief :  Routine to configure aic3104 to loop Mic to head-phone
 *
 * \b Descr: This routine initializes the aic3104 Audio Codec on the
 * \n        base board to take input from the microphone and play it
 * \n        back on the headphones.
 *
 * \param :u8McaspNum	[IN]	Instance of McASP number
 *
 * \return:  SUCCESS for success  - Description
 * \n        FAILED for error   - Description
 */

STATUS aic3104_MicHeadphone_Loop
(
	UINT8    u8McaspNum
);

/***
 * \brief: Routine to test the aic3104 Audio codec on Base board
 *
 * \b Descr: This routine initializes the aic3104 Audio Codec on the
 * \n        base board to test various interfaces including the
 * \n        following.
 *
 * \n      (a) Playback on Head-phohne
 * \n      (b) Playback on Line-Out
 * \n      (c) Loopback the input on Microphone to head-set.
 * \n      (d) Loopback the input on Line-In to Line-Out
 *
 * \param: u8McaspNum	[IN]	Instance of McASP number
 *
 * \return   SUCCESS    success
 * \n        FAILED     error
 */
STATUS aic3104_tests
(
	UINT8    u8McaspNum
);

/***
 * \brief:   Routine to test the AIC3104 Audio codec on Base board
 *
 * \b Descr: This routine initializes the AIC3104 Audio Codec on the base board
 * \n        to test various interfaces including the following.
 *
 *	\n      (a) Playback on Head-phohne
 *  \n      (b) Playback on Line-Out
 *  \n      (c) Loopback the input on Microphone to head-set.
 *  \n      (d) Loopback the input on Line-In to Line-Out
 *
 * \param:  u8McaspNum	[IN]	Instance of McASP number
 *
 * \return:    SUCCESS for success  - Description
 * \n 	       FAILED for error   - Description
 *
 */
STATUS Audio_Test
(
	void *testargs
);

/* @} */   /* End of aic3104 Audio Codec */

#ifdef __cplusplus
} /* End of extern C */
#endif /* #ifdef __cplusplus */

#endif /* _DM8127_aic3104_H_ */
